<?php
/* --------------------------------------------------------------
   ProcessedImageInformationRepository.php 2023-02-01
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2023 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\ImageProcessing\App\Data\Gd;

use Gambio\Admin\Modules\ImageProcessing\Exceptions\ImageNotFoundException;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\CustomImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\Configuration\ImageConfiguration;
use Gambio\Admin\Modules\ImageProcessing\Model\ProcessedImageInformation;

/**
 * Class ProcessedImageInformationRepository
 *
 * @package Gambio\Admin\Modules\ImageProcessing\App\Data
 */
class ProcessedImageInformationRepository
{
    /**
     * Returns information about the processed image based on the image configuration.
     *
     * @param string             $source
     * @param string             $destination
     * @param ImageConfiguration $imageConfiguration
     *
     * @return ProcessedImageInformation
     * @throws ImageNotFoundException
     */
    public function getProcessedImageInformation(
        string             $source,
        string             $destination,
        ImageConfiguration $imageConfiguration
    ): ProcessedImageInformation {
        $imageInfo = getimagesize($source);
        if ($imageInfo === false) {
            $message = "Image in path '$source' not found.";
            throw new ImageNotFoundException($message);
        }
        
        [$width, $height, $type] = $imageInfo;
        if ($imageConfiguration->width() === 0) {
            $heightRatio = $height / $imageConfiguration->height();
            $maxWidth    = $width / $heightRatio;
        } else {
            $maxWidth = $imageConfiguration->width();
        }
        $widthRatio  = $width / $imageConfiguration->width();
        $heightRatio = $height / $imageConfiguration->height();
        $finalWidth  = $widthRatio > $heightRatio ? $maxWidth : (int)round($width / $heightRatio);
        $finalHeight = $widthRatio > $heightRatio ? (int)round($height / $widthRatio) : $imageConfiguration->height();
        
        return new ProcessedImageInformation($source,
                                             $destination,
                                             $maxWidth,
                                             $imageConfiguration->height(),
                                             $width,
                                             $height,
                                             $type,
                                             $finalWidth,
                                             $finalHeight);
    }
    
    
    /**
     * Returns information about the processed image based on the custom image configuration.
     *
     * The difference between `::getProcessedImageInformation` is that the width and height
     * of the processed image taken from the $imageConfiguration instead of calculating the
     * ratio.
     *
     * @param string                   $sourceFile
     * @param string                   $destinationFile
     * @param CustomImageConfiguration $imageConfiguration
     *
     * @return ProcessedImageInformation
     * @throws ImageNotFoundException
     * @see ProcessedImageInformationRepository::getProcessedImageInformation
     */
    public function getCustomProcessedImageConfiguration(
        string                   $sourceFile,
        string                   $destinationFile,
        CustomImageConfiguration $imageConfiguration
    ): ProcessedImageInformation {
        $imageInfo = getimagesize($sourceFile);
        if ($imageInfo === false) {
            $message = "Image in path '$sourceFile' not found.";
            throw new ImageNotFoundException($message);
        }
        
        [$width, $height, $type] = $imageInfo;
        
        $maxWidth    = $imageConfiguration->width();
        $maxHeight   = $imageConfiguration->height();
        $imageWidth  = $width;
        $imageHeight = $height;
        $imageType   = $type;
        $finalWidth  = $imageConfiguration->width();
        $finalHeight = $imageConfiguration->height();
        
        return new ProcessedImageInformation($sourceFile,
                                             $destinationFile,
                                             $maxWidth,
                                             $maxHeight,
                                             $imageWidth,
                                             $imageHeight,
                                             $imageType,
                                             $finalWidth,
                                             $finalHeight);
    }
}